// main.js - v2 - Added cache busting for worker

const countdownOverlay = document.getElementById('countdownOverlay');
const tabir = document.getElementById('tabir');
const countdownOverlayText = document.getElementById('countdownOverlayText');
const countdownCircleCanvas = document.getElementById('countdownCircleCanvas');
const ctx = countdownCircleCanvas.getContext('2d');
let countdownOverlayVisible = false;

// Global worker references
let clockWorker;
let prayerWorker;
let slideWorker;

function drawProgressBar(percentage) {
    if (!ctx) return;
    const size = countdownCircleCanvas.width;
    const radius = size / 2 - 10;
    const centerX = size / 2;
    const centerY = size / 2;
    const startAngle = -Math.PI / 2;
    const endAngle = startAngle + (Math.PI * 2 * (percentage / 100));

    ctx.clearRect(0, 0, size, size);

    // Background circle
    ctx.beginPath();
    ctx.arc(centerX, centerY, radius, 0, Math.PI * 2);
    ctx.lineWidth = 10;
    ctx.strokeStyle = 'rgba(255, 255, 255, 0.2)';
    ctx.setLineDash([5, 5]);
    ctx.stroke();

    // Progress bar
    ctx.beginPath();
    ctx.arc(centerX, centerY, radius, startAngle, endAngle);
    ctx.lineWidth = 10;
    ctx.lineCap = 'round';
    ctx.setLineDash([]);
    const gradient = ctx.createLinearGradient(0, 0, size, size);
    gradient.addColorStop(0, '#f59e0b');
    gradient.addColorStop(1, '#ef4444');
    ctx.strokeStyle = gradient;
    ctx.stroke();
}

function initializeCountdownCanvas() {
    const container = countdownCircleCanvas.parentElement;
    if (!container) return;
    const size = Math.min(container.offsetWidth, container.offsetHeight);
    countdownCircleCanvas.width = size;
    countdownCircleCanvas.height = size;
    if (countdownOverlayVisible) {
        drawProgressBar(0); // Redraw if visible
    }
}

function initializeMainDisplay() {
    const clockOffset = (typeof config !== 'undefined' && config.timeOffset) ? config.timeOffset * 1000 : 0;
    console.log('🚀 Initializing Main Display with Cache-Busted Unified Worker');

    // --- UNIFIED CLOCK/COUNTDOWN WORKER WITH CACHE BUSTING ---
    const workerUrl = 'combined_clock.js?v=' + new Date().getTime();
    clockWorker = new Worker(workerUrl);
    const prayerOrder = ['subuh', 'dhuhur', 'ashar', 'maghrib', 'isya'];
    clockWorker.postMessage({ 
        clockOffset, 
        prayerTimes, 
        prayerOrder
    });

    clockWorker.onmessage = (e) => {
        const { time, countdown, label, second, totalSeconds } = e.data;

        // 1. Update Digital Clock
        const clockElement = document.getElementById('digitalClock');
        if (clockElement) {
            clockElement.innerHTML = time;
        }

        // 2. Update Countdown Timer
        const countdownElement = document.getElementById('countdown');
        const labelElement = document.getElementById('nextPrayerLabel');
        if (countdownElement) {
            countdownElement.textContent = countdown;
        }
        if (labelElement) {
            labelElement.textContent = label.split(' ')[0];
        }

        // 3. Handle Countdown Overlay
        if (totalSeconds <= 60 && totalSeconds > 0) {
            if (!countdownOverlayVisible) {
                countdownOverlay.classList.add('show');
                countdownOverlayVisible = true;
                initializeCountdownCanvas();
            }
            countdownOverlayText.textContent = totalSeconds.toString();
            const percentage = ((60 - totalSeconds) / 60) * 100;
            drawProgressBar(percentage);
        } else {
            if (countdownOverlayVisible) {
                countdownOverlay.classList.remove('show');
                countdownOverlayVisible = false;
            }
        }

        // 4. Handle Adzan Redirect
        if (totalSeconds === 0) {
            if (clockWorker) clockWorker.terminate();
            if (prayerWorker) prayerWorker.terminate();
            if (slideWorker) slideWorker.terminate();
            window.location.href = 'adzan-display.php';
        }

        // 5. Trigger other updates on the 0th second
        if (second === 0) {
            console.log("Minute tick: Running scheduled updates.");
            updateDateInfo();
            manageVideoStream();
            if (slideWorker) slideWorker.postMessage('get');
            
            const now = new Date(Date.now() + clockOffset);
            if (now.getMinutes() === 0) {
                console.log("Hourly tick: Refreshing page for cleanup.");
                location.reload();
            }
        }
    };

    // --- OTHER WORKERS (UNCHANGED) ---
    prayerWorker = new Worker('prayer.js');
    prayerWorker.postMessage({ dailySchedule, prayerTimes, clockOffset });
    prayerWorker.onmessage = (e) => {
        updatePrayerHighlight(e.data.current, e.data.next);
    };

    slideWorker = new Worker('slide.js');
    slideWorker.postMessage({ manifest: slideshowManifest, prayerTimes, config, clockOffset });
    slideWorker.onmessage = (e) => {
        const imageUrl = e.data;
        showImageSlide(imageUrl);
    };
    
    // --- UI AND OTHER FUNCTIONS ---
    function updateDateInfo() {
        const dateElement = document.getElementById('currentDate');
        if (!dateElement) return;
        const now = new Date(Date.now() + clockOffset);
        const days = ['Minggu', 'Senin', 'Selasa', 'Rabu', 'Kamis', 'Jumat', 'Sabtu'];
        const months = ['Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni', 'Juli', 'Agustus', 'September', 'Oktober', 'November', 'Desember'];
        dateElement.textContent = `${days[now.getDay()]}, ${now.getDate()} ${months[now.getMonth()]} ${now.getFullYear()}`;
    }

    function updatePrayerHighlight(currentPrayer, nextPrayer) {
        document.querySelectorAll('.prayer-item').forEach(item => {
            item.classList.remove('current-prayer', 'next-prayer');
        });
        if (currentPrayer) {
            const currentElement = document.querySelector(`[data-prayer="${currentPrayer}"]`);
            if (currentElement) currentElement.classList.add('current-prayer');
        }
        if (nextPrayer) {
            const nextElement = document.querySelector(`[data-prayer="${nextPrayer}"]`);
            if (nextElement) nextElement.classList.add('next-prayer');
        }
    }

    function initRunningText() {
        const container = document.getElementById('runningText');
        if (!container || typeof runningTextArray === 'undefined' || !runningTextArray.length) return;
        const shuffledText = [...runningTextArray].sort(() => Math.random() - 0.5).slice(0, 5);
        const separator = '   •   ';
        container.innerHTML = `<div class="running-text-wrapper"><span>${shuffledText.join(separator) + separator}</span><span>${shuffledText.join(separator) + separator}</span></div>`;
    }

    const slideshowContainer = document.getElementById('slideshowContainer');
    let isShowingVideo = false;

    function showImageSlide(imageUrl) {
        if (!slideshowContainer || isShowingVideo) return;
        slideshowContainer.innerHTML = `<img src="${imageUrl}" class="slideshow-container__image" alt="Slideshow Image">`;
    }

    function showVideoStream() {
        if (!slideshowContainer) return;
        slideshowContainer.innerHTML = `<video src="slideshow/kabah.mp4" style="width:100%;height:100%;object-fit:cover;" autoplay muted loop playsinline></video>`;
    }

    function manageVideoStream() {
        const now = new Date(Date.now() + clockOffset);
        const interval = (typeof config !== 'undefined' && config.youtubeIntervalMinutes) ? config.youtubeIntervalMinutes : 10;
        const shouldShowVideo = now.getMinutes() % interval === 0;

        if (shouldShowVideo && !isShowingVideo) {
            isShowingVideo = true;
            showVideoStream();
        } else if (!shouldShowVideo && isShowingVideo) {
            isShowingVideo = false;
        }
    }

    // Initial setup calls
    updateDateInfo();
    initRunningText();
    window.addEventListener('resize', initializeCountdownCanvas);
    initializeCountdownCanvas();
    manageVideoStream();
}

document.addEventListener("DOMContentLoaded", () => {
    const tabir = document.getElementById("tabir");
    if (tabir) tabir.style.display = "none";

    setTimeout(() => {
        if (document.getElementById('slideshowContainer')) {
            initializeMainDisplay();
        }
    }, 500); 
});
