<?php
// index.php - Main container for iframe loading main.php and other displays with server failover
// =======================================================================

require_once 'conf.php';
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <link rel="shortcut icon" href="favicon.ico" type="image/x-icon">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo "$masjid_name"; ?> - Tampilan Utama</title>
    <style>
        html, body {
            margin: 0;
            padding: 0;
            width: 100%;
            height: 100%;
            overflow: hidden;
            cursor: none;
            border: none;
        }
        body {
            position: relative;
            background-color: #000000;
            container-type: size;
        }
        #content-container {
            border: none;
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            aspect-ratio: 16 / 9;
            background-color: #000000;
            display: flex;
            justify-content: center;
            align-items: center;
            text-align: center;
        }
        @container (min-aspect-ratio: 16 / 9) {
            #content-container {
                height: 100%;
            }
        }
        @container (max-aspect-ratio: 16 / 9) {
            #content-container {
                width: 100%;
            }
        }
        #mainFrame {
            width: 100%;
            height: 100%;
            border: none;
            display: block;
            cursor: none;
            background-color: #000000;
        }
        #preload { display: none; }
        #connection-indicator {
            position: fixed;
            top: 15px;
            right: 15px;
            width: 15px;
            height: 15px;
            border-radius: 50%;
            background-color: #ffc107;
            box-shadow: 0 0 8px rgba(0, 0, 0, 0.5);
            transition: background-color 0.5s ease;
            z-index: 9999;
    	    opacity: 0.8;
            display: none;
        }
        #connection-indicator.status-good {
            background-color: #28a745; /* Hijau */
        }
        #connection-indicator.status-degraded {
            background-color: #ffc107; /* Oranye */
        }
        #connection-indicator.status-disconnected {
            background-color: #dc3545; /* Merah */
        }
    </style>
</head>
<body>

    <div id="connection-indicator" title="Status Koneksi"></div>
    <div id="content-container">
        <iframe id="mainFrame" src="main.php" frameborder="0" allowfullscreen></iframe>
    </div>

<script>
document.addEventListener('DOMContentLoaded', () => {

    const indicator = document.getElementById('connection-indicator');
    const mainFrame = document.getElementById('mainFrame');

    const PING_URL = 'ping.txt';
    const GOOD_THRESHOLD = 2500; 
    const DEGRADED_THRESHOLD = 20000; // THE FIX: Increased to 20 seconds
    
    let lastStatus = null;

    function updateIndicator(status) {
        if (status === lastStatus) return;
        indicator.classList.remove('status-good', 'status-degraded', 'status-disconnected');
        indicator.classList.add(`status-${status}`);
        lastStatus = status;
    }

    async function checkConnection() {
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), DEGRADED_THRESHOLD);
        const startTime = performance.now();
        try {
            const response = await fetch(`${PING_URL}?t=${new Date().getTime()}`, {
                method: 'HEAD',
                cache: 'no-store',
                signal: controller.signal
            });
            clearTimeout(timeoutId);
            const latency = performance.now() - startTime;
            if (response.ok && latency < GOOD_THRESHOLD) {
                return 'good';
            } else {
                return 'degraded';
            }
        } catch (error) {
            clearTimeout(timeoutId);
            return 'disconnected';
        }
    }

    // THE FIX: Non-aggressive initial check
    setTimeout(async () => {
        const initialStatus = await checkConnection();
        if (initialStatus === 'disconnected') {
            console.log("Initial connection check failed, but will not force a reload to prevent race conditions.");
        }
    }, 1000);

});
</script>

</body>
</html>
