// adzan-layer.js - Adzan Layer Management for Main.php
// ====================================================

class AdzanLayerManager {
    constructor() {
        console.log("AdzanLayerManager constructor called");
        console.log("Initializing elements...");
        
        this.elements = {
            progressBar: document.getElementById('progressBar'),
            progressText: document.getElementById('progressText'),
            prayerName: document.getElementById('prayerName'),
            adzanTitle: document.getElementById('adzanTitle'),
            arabicMessage: document.getElementById('arabicMessage'),
            indonesianMessage: document.getElementById('indonesianMessage')
        };
        
        console.log("Elements found:", {
            progressBar: !!this.elements.progressBar,
            progressText: !!this.elements.progressText,
            prayerName: !!this.elements.prayerName,
            adzanTitle: !!this.elements.adzanTitle,
            arabicMessage: !!this.elements.arabicMessage,
            indonesianMessage: !!this.elements.indonesianMessage
        });
        
        this.isActive = false;
        this.currentState = null;
    }

    // Check if the layer is currently active
    isCurrentlyActive() {
        return this.isActive;
    }

    // Get current state
    getCurrentState() {
        return this.currentState;
    }

    // Activate adzan layer with state data
    activate(state) {
        console.log("Activating adzan layer with state:", state);
        this.isActive = true;
        this.currentState = state;
        this.updateDisplay(state);
        console.log('Adzan layer activated for:', state.prayer);
    }

    // Deactivate adzan layer
    deactivate() {
        this.isActive = false;
        this.currentState = null;
        console.log('Adzan layer deactivated');
    }

    // Update display elements based on state
    updateDisplay(state) {
        if (!this.isActive || !state) {
            console.log("Update skipped - not active or no state");
            return;
        }

        console.log("Updating display with state:", state);

        // Update prayer name
        if (this.elements.prayerName) {
            this.elements.prayerName.textContent = state.prayer || 'Memuat...';
            console.log("Updated prayer name:", state.prayer);
        }

        // Special handling for Shuruq
        const isShuruq = state.prayer && state.prayer.toLowerCase() === 'shuruq';
        
        if (this.elements.adzanTitle) {
            this.elements.adzanTitle.style.display = isShuruq ? 'none' : 'block';
        }
        
        if (this.elements.arabicMessage) {
            this.elements.arabicMessage.style.display = isShuruq ? 'none' : 'block';
        }
        
        if (this.elements.indonesianMessage) {
            this.elements.indonesianMessage.style.display = isShuruq ? 'none' : 'block';
        }

        if (this.elements.prayerName && isShuruq) {
            this.elements.prayerName.style.fontSize = '8rem';
            this.elements.prayerName.style.marginTop = '2rem';
        } else if (this.elements.prayerName) {
            this.elements.prayerName.style.fontSize = '';
            this.elements.prayerName.style.marginTop = '';
        }

        // Update progress bar
        if (this.elements.progressBar && typeof state.progress === 'number') {
            this.elements.progressBar.style.width = state.progress + '%';
            console.log("Updated progress bar:", state.progress + '%');
        }

        // Update countdown text
        if (state.remainingTime !== undefined && this.elements.progressText) {
            const minutes = Math.floor(state.remainingTime / 60);
            const seconds = Math.floor(state.remainingTime % 60);
            const timeText = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
            this.elements.progressText.textContent = timeText;
            console.log("Updated countdown text:", timeText);
        }

        // Update progress bar color for urgency
        if (state.remainingTime <= 30) {
            if (this.elements.progressBar) {
                this.elements.progressBar.style.background = 
                    'linear-gradient(90deg, #ef4444 0%, #dc2626 50%, #b91c1c 100%)';
            }
            if (this.elements.progressText) {
                this.elements.progressText.style.color = '#fecaca';
            }
        } else {
            if (this.elements.progressBar) {
                this.elements.progressBar.style.background = '';
            }
            if (this.elements.progressText) {
                this.elements.progressText.style.color = '';
            }
        }
    }

    // Handle state updates from event worker
    handleStateUpdate(state) {
        if (!this.isActive) {
            console.log("State update ignored - layer not active");
            return;
        }
        
        console.log("Handling state update:", state);
        this.updateDisplay(state);

        // Check if we should transition to next phase
        if (state.remainingTime <= 0) {
            console.log('Adzan phase complete, should transition to iqomah');
            // Let the main layer manager handle the transition
            return 'transition_to_iqomah';
        }

        return 'continue';
    }
}

// Export for use in main.js
window.AdzanLayerManager = AdzanLayerManager;
