// iqomah-layer.js - Iqomah Layer Management for Main.php
// ======================================================

class IqomahLayerManager {
    constructor() {
        console.log("IqomahLayerManager constructor called");
        console.log("Initializing elements...");
        
        this.elements = {
            prayerInfo: document.getElementById('prayerInfo'),
            countdownTime: document.getElementById('countdownTime'),
            nasihatItem: document.getElementById('nasihatItem')
        };
        
        console.log("Elements found:", {
            prayerInfo: !!this.elements.prayerInfo,
            countdownTime: !!this.elements.countdownTime,
            nasihatItem: !!this.elements.nasihatItem
        });
        
        this.isActive = false;
        this.currentState = null;
        this.currentNasihatIndex = 0;
        this.nasihatInterval = null;
        
        // Initialize nasihat rotation if elements exist
        if (this.elements.nasihatItem) {
            this.initializeNasihatRotation();
        }
    }
    
    initializeNasihatRotation() {
        if (!window.nasihatList || window.nasihatList.length === 0) {
            console.log("No nasihat list available");
            return;
        }
        
        this.updateNasihat();
        this.nasihatInterval = setInterval(() => {
            this.updateNasihat();
        }, 8000);
    }
    
    updateNasihat() {
        if (!window.nasihatList || window.nasihatList.length === 0 || !this.elements.nasihatItem) {
            return;
        }
        
        this.elements.nasihatItem.classList.remove('active');
        setTimeout(() => {
            this.currentNasihatIndex = (this.currentNasihatIndex + 1) % window.nasihatList.length;
            this.elements.nasihatItem.textContent = window.nasihatList[this.currentNasihatIndex];
            this.elements.nasihatItem.classList.add('active');
        }, 500);
    }
    
    activate(state) {
        if (!state) {
            console.log("No state provided to IqomahLayerManager");
            return;
        }
        
        console.log("Activating iqomah layer with state:", state);
        this.isActive = true;
        this.currentState = state;
        
        // Update prayer name
        if (this.elements.prayerInfo) {
            this.elements.prayerInfo.textContent = state.prayer || "Iqomah";
        }
        
        // Update countdown
        this.updateCountdown(state);
        
        // Update body classes for visual effects
        this.updateBodyClasses(state);
        
        // Check for completion
        if (state.remainingTime <= 0) {
            console.log("Iqomah countdown finished, redirecting to blank screen");
            window.location.href = "blank-screen.php";
            return;
        }
    }
    
    updateCountdown(state) {
        if (!this.elements.countdownTime || !state) return;
        
        const minutes = Math.floor(state.remainingTime / 60);
        const seconds = Math.floor(state.remainingTime % 60);
        const timeText = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
        
        this.elements.countdownTime.textContent = timeText;
    }
    
    updateBodyClasses(state) {
        const body = document.body;
        if (!body || !state) return;
        
        // Remove existing mode classes
        body.classList.remove('urgent-mode', 'beep-mode');
        
        // Add appropriate classes based on remaining time
        if (state.remainingTime <= 20 && state.remainingTime > 5) {
            body.classList.add('urgent-mode');
        } else if (state.remainingTime <= 5 && state.remainingTime > 0) {
            body.classList.add('beep-mode');
        }
    }
    
    deactivate() {
        console.log("Deactivating iqomah layer");
        this.isActive = false;
        this.currentState = null;
        
        // Clear nasihat interval
        if (this.nasihatInterval) {
            clearInterval(this.nasihatInterval);
            this.nasihatInterval = null;
        }
        
        // Remove body classes
        const body = document.body;
        if (body) {
            body.classList.remove('urgent-mode', 'beep-mode');
        }
        
        // Reset elements
        if (this.elements.prayerInfo) {
            this.elements.prayerInfo.textContent = "Memuat...";
        }
        if (this.elements.countdownTime) {
            this.elements.countdownTime.textContent = "--:--";
        }
    }
    
    isCurrentlyActive() {
        return this.isActive;
    }
    
    getCurrentState() {
        return this.currentState;
    }
}

// Make IqomahLayerManager available globally
window.IqomahLayerManager = IqomahLayerManager;

console.log("IqomahLayerManager class defined and made available globally");
