// main.js - v3 - Integrated Iqomah Layer

const countdownOverlay = document.getElementById('countdownOverlay');
const tabir = document.getElementById('tabir');
const countdownOverlayText = document.getElementById('countdownOverlayText');
const countdownCircleCanvas = document.getElementById('countdownCircleCanvas');
const ctx = countdownCircleCanvas.getContext('2d');
let countdownOverlayVisible = false;

// Global worker references
let clockWorker;
let prayerWorker;
let slideWorker;
let eventWorker;

// Additional state variables
let adzanLayerManager = null;
let iqomahLayerManager = null; // Added for iqomah
let currentActiveLayer = null;
const slideshowContainer = document.getElementById("slideshowContainer");
let isShowingVideo = false;
let currentImageUrl = "";
let preloadedImage = null;
let currentVideoElement = null;
let workersState = {
    paused: false,
    slideWorkerActive: true
};

function initializeMainDisplay() {
    const clockOffset = (typeof config !== 'undefined' && config.timeOffset) ? config.timeOffset * 1000 : 0;
    console.log('🚀 Initializing Main Display with Cache-Busted Unified Worker');

    // Initialize workers
    initializeClockWorker(clockOffset);
    initializePrayerWorker(clockOffset);
    initializeSlideWorker(clockOffset);
    initializeEventWorker(clockOffset);
    
    // Initial setup calls
    updateDateInfo();
    initRunningText();
    window.addEventListener("resize", initializeCountdownCanvas);
    initializeCountdownCanvas();
    manageVideoStream();
}

function initializeClockWorker(clockOffset) {
    const workerUrl = 'combined_clock.js?v=' + new Date().getTime();
    clockWorker = new Worker(workerUrl);
    clockWorker.postMessage({
        clockOffset,
        prayerTimes,
        prayerOrder: ["subuh", "dhuhur", "jumatan", "ashar", "maghrib", "isya"]
    });

    clockWorker.onmessage = (e) => {
        const { time, countdown, label, second, totalSeconds } = e.data;
        updateDisplay({ time, countdown, label, totalSeconds });
        
        // Trigger other updates on the 0th second
        if (second === 0) {
            console.log("Minute tick: Running scheduled updates.");
            updateDateInfo();
            manageVideoStream();
            if (slideWorker) slideWorker.postMessage("get");
        }
    };
}

function initializePrayerWorker(clockOffset) {
    prayerWorker = new Worker('prayer.js');
    prayerWorker.postMessage({ dailySchedule, prayerTimes, clockOffset });
    prayerWorker.onmessage = (e) => {
        updatePrayerHighlight(e.data.current, e.data.next);
    };
}

function initializeSlideWorker(clockOffset) {
    slideWorker = new Worker('slide.js');
    slideWorker.postMessage({ manifest: slideshowManifest, prayerTimes, config, clockOffset });
    slideWorker.onmessage = (e) => {
        const imageUrl = e.data;
        showImageSlide(imageUrl);
    };
}

function initializeEventWorker(clockOffset) {
    eventWorker = new Worker('event.js');
    eventWorker.postMessage({ dailySchedule, config, clockOffset });

    eventWorker.onmessage = (e) => {
        const state = e.data;

        // Determine the active mode, defaulting to 'default'
        const activeMode = state.isActive ? state.mode : 'default';

        // Switch the visual layer only if the mode has changed
        if (activeMode !== currentActiveLayer) {
            toggleLayer(activeMode);
        }

        // Always pass the state to the relevant manager if its layer is active
        if (state.isActive) {
            if (state.mode === 'adzan' && adzanLayerManager) {
                adzanLayerManager.activate(state);
            } else if (state.mode === 'iqomah' && iqomahLayerManager) {
                iqomahLayerManager.activate(state);
            }
        }
    };
}

function toggleLayer(mode) {
    console.log(`Switching to layer: ${mode}`);
    currentActiveLayer = mode; // Update the current logical layer

    // Deactivate managers if their layer is no longer active
    if (mode !== 'adzan' && adzanLayerManager && adzanLayerManager.isCurrentlyActive()) {
        adzanLayerManager.deactivate();
    }
    if (mode !== 'iqomah' && iqomahLayerManager && iqomahLayerManager.isCurrentlyActive()) {
        iqomahLayerManager.deactivate();
    }

    const layers = {
        adzan: "adzanOverlay",
        iqomah: "iqomahOverlay",
        sholat: "blankOverlay",
        blank: "blankOverlay"
    };

    // Hide all layer elements first
    Object.values(layers).forEach(layerId => {
        const element = document.getElementById(layerId);
        if (element) {
            element.classList.remove("active");
            element.style.display = "none";
        }
    });

    const runningTextElement = document.getElementById("runningText");
    const mainContent = document.getElementById("main-content");

    if (mode === 'default') {
        if (mainContent) mainContent.style.display = 'block';
        if (runningTextElement) runningTextElement.style.display = 'block';

        // Resume slideshow if it was paused
        if (slideWorker && workersState.paused) {
            slideWorker.postMessage("get");
            workersState.paused = false;
        }
        console.log("Returned to default view");
        return;
    }

    // For any other mode, hide the main content and pause background tasks
    if (mainContent) mainContent.style.display = 'none';
    if (runningTextElement) runningTextElement.style.display = 'none';
    if (slideWorker && !workersState.paused) {
        workersState.paused = true;
    }

    const targetLayerId = layers[mode];
    if (targetLayerId) {
        const targetElement = document.getElementById(targetLayerId);
        if (targetElement) {
            targetElement.style.display = 'block';
            requestAnimationFrame(() => {
                targetElement.classList.add('active');
                console.log(`Activated visual layer: ${targetLayerId}`);
            });
        } else {
            console.error(`Target element not found: ${targetLayerId}`);
        }
    }
}


function updateDisplay(data) {
    const { time, countdown, label, totalSeconds } = data;
    const clockElement = document.getElementById('digitalClock');
    const countdownElement = document.getElementById('countdown');
    const labelElement = document.getElementById('nextPrayerLabel');

    if (clockElement) {
        clockElement.innerHTML = time;
    }
    if (countdownElement) {
        countdownElement.textContent = countdown;
    }
    if (labelElement) {
        labelElement.textContent = label.split(' ')[0];
    }
    handleCountdownOverlay(totalSeconds);
}

function handleCountdownOverlay(totalSeconds) {
    if (totalSeconds <= 60 && totalSeconds > 0) {
        if (!countdownOverlayVisible) {
            countdownOverlay.classList.add('show');
            countdownOverlayVisible = true;
            initializeCountdownCanvas();
        }
        countdownOverlayText.textContent = totalSeconds.toString();
        const percentage = ((60 - totalSeconds) / 60) * 100;
        drawProgressBar(percentage);
    } else if (countdownOverlayVisible) {
        countdownOverlay.classList.remove('show');
        countdownOverlayVisible = false;
    }
}

function updatePrayerHighlight(currentPrayer, nextPrayer) {
    document.querySelectorAll('.prayer-item').forEach(item => {
        item.classList.remove('current-prayer', 'next-prayer');
    });
    if (currentPrayer) {
        const currentElement = document.querySelector(`[data-prayer="${currentPrayer}"]`);
        if (currentElement) currentElement.classList.add('current-prayer');
    }
    if (nextPrayer) {
        const nextElement = document.querySelector(`[data-prayer="${nextPrayer}"]`);
        if (nextElement) nextElement.classList.add('next-prayer');
    }
}

function showImageSlide(imageUrl) {
    if (!slideshowContainer || imageUrl === currentImageUrl) return;

    try {
        slideshowContainer.classList.add("slideshow-container__loading");

        const newImageElement = document.createElement("img");
        newImageElement.src = imageUrl;
        newImageElement.className = "slideshow-container__image incoming";
        newImageElement.alt = "Slideshow Image";

        const currentImage = slideshowContainer.querySelector(".slideshow-container__image");

        if (currentImage) {
            currentImage.style.transition = "opacity 2.5s ease-out, transform 2.5s ease-out";
            currentImage.style.zIndex = "2";
            
            newImageElement.style.opacity = "0";
            newImageElement.style.transform = "scale(1.05)";
            newImageElement.style.transition = "opacity 2.5s ease-out, transform 2.5s ease-out";
            newImageElement.style.position = "absolute";
            newImageElement.style.top = "0";
            newImageElement.style.left = "0";
            newImageElement.style.width = "100%";
            newImageElement.style.height = "100%";
            newImageElement.style.objectFit = "cover";
            newImageElement.style.zIndex = "3";

            slideshowContainer.appendChild(newImageElement);
            
            setTimeout(() => {
                newImageElement.style.opacity = "1";
                newImageElement.style.transform = "scale(1)";
                currentImage.style.opacity = "0";
                currentImage.style.transform = "scale(0.95)";
            }, 50);

            setTimeout(() => {
                if (currentImage.parentNode) {
                    currentImage.remove();
                }
                newImageElement.style.position = "static";
                newImageElement.style.zIndex = "auto";
                newImageElement.className = "slideshow-container__image";
            }, 2600);

        } else {
            slideshowContainer.innerHTML = ""; // Clear everything including video
            newImageElement.style.opacity = "0";
            newImageElement.style.transition = "opacity 1s ease-in-out";
            slideshowContainer.appendChild(newImageElement);

            setTimeout(() => {
                newImageElement.style.opacity = "1";
                newImageElement.className = "slideshow-container__image";
            }, 100);
        }

        slideshowContainer.classList.remove("slideshow-container__loading");
        currentImageUrl = imageUrl;

    } catch (error) {
        console.warn("Failed to load image:", imageUrl, error);
        slideshowContainer.innerHTML = `<img src="${imageUrl}" class="slideshow-container__image" alt="Slideshow Image">`;
        slideshowContainer.classList.remove("slideshow-container__loading");
    }
}

function initRunningText() {
    const container = document.getElementById('runningText');
    if (!container || typeof runningTextArray === 'undefined' || !runningTextArray.length) return;
    const shuffledText = [...runningTextArray].sort(() => Math.random() - 0.5).slice(0, 5);
    const separator = '   •   ';
    container.innerHTML = `<div class="running-text-wrapper"><span>${shuffledText.join(separator) + separator}</span><span>${shuffledText.join(separator) + separator}</span></div>`;
}

function initializeCountdownCanvas() {
    const container = countdownCircleCanvas.parentElement;
    if (!container) return;
    const size = Math.min(container.offsetWidth, container.offsetHeight);
    countdownCircleCanvas.width = size;
    countdownCircleCanvas.height = size;
    if (countdownOverlayVisible) {
        drawProgressBar(0); // Redraw if visible
    }
}

function drawProgressBar(percentage) {
    if (!ctx) return;
    const size = countdownCircleCanvas.width;
    const radius = size / 2 - 10;
    const centerX = size / 2;
    const centerY = size / 2;
    const startAngle = -Math.PI / 2;
    const endAngle = startAngle + (Math.PI * 2 * (percentage / 100));

    ctx.clearRect(0, 0, size, size);

    // Background circle
    ctx.beginPath();
    ctx.arc(centerX, centerY, radius, 0, Math.PI * 2);
    ctx.lineWidth = 10;
    ctx.strokeStyle = 'rgba(255, 255, 255, 0.2)';
    ctx.setLineDash([5, 5]);
    ctx.stroke();

    // Progress bar
    ctx.beginPath();
    ctx.arc(centerX, centerY, radius, startAngle, endAngle);
    ctx.lineWidth = 10;
    ctx.lineCap = 'round';
    ctx.setLineDash([]);
    const gradient = ctx.createLinearGradient(0, 0, size, size);
    gradient.addColorStop(0, '#f59e0b');
    gradient.addColorStop(1, '#ef4444');
    ctx.strokeStyle = gradient;
    ctx.stroke();
}

// Initialize once DOM is ready
document.addEventListener("DOMContentLoaded", () => {
    const tabir = document.getElementById("tabir");
    if (tabir) tabir.style.display = "none";
    setTimeout(() => {
        adzanLayerManager = new window.AdzanLayerManager();
        iqomahLayerManager = new window.IqomahLayerManager(); // Instantiated iqomah layer manager
        initializeMainDisplay();
    }, 500); // Ensure DOM is fully loaded and ready
});
function updateDateInfo() {
    const dateElement = document.getElementById('currentDate');
    if (!dateElement) return;
    const now = new Date();
    const days = ['Minggu', 'Senin', 'Selasa', 'Rabu', 'Kamis', 'Jumat', 'Sabtu'];
    const months = ['Januari', 'Februari', 'Maret', 'April', 'Mei', 'Juni', 'Juli', 'Agustus', 'September', 'Oktober', 'November', 'Desember'];
    dateElement.textContent = `${days[now.getDay()]}, ${now.getDate()} ${months[now.getMonth()]} ${now.getFullYear()}`;
}

function manageVideoStream() {
    const now = new Date();
    const interval = (typeof config !== "undefined" && config.youtubeIntervalMinutes) ? config.youtubeIntervalMinutes : 10;
    const shouldShowVideo = now.getMinutes() % interval === 0;

    if (shouldShowVideo && !isShowingVideo) {
        showVideoStream();
    } else if (!shouldShowVideo && isShowingVideo) {
        stopVideoStream();
    }
}

function stopVideoStream() {
    if (currentVideoElement) {
        currentVideoElement.pause();
        currentVideoElement.removeAttribute("src");
        currentVideoElement.load();
        currentVideoElement = null;
    }
    isShowingVideo = false;
    console.log("Video stream stopped and cleaned up");
}

function showVideoStream() {
    if (!slideshowContainer) return;
    
    stopVideoStream();
    slideshowContainer.innerHTML = "";
    
    const videoElement = document.createElement("video");
    videoElement.src = "slideshow/kabah.mp4";
    videoElement.style.cssText = "width:100%;height:100%;object-fit:cover;";
    videoElement.autoplay = true;
    videoElement.muted = true;
    videoElement.loop = true;
    videoElement.playsInline = true;
    
    currentVideoElement = videoElement;
    slideshowContainer.appendChild(videoElement);
    isShowingVideo = true;
    
    console.log("Video stream started");
}

function preloadImage(url) {
    return new Promise((resolve, reject) => {
        const img = new Image();
        img.onload = () => resolve(img);
        img.onerror = reject;
        img.src = url;
    });
}
