// ==========================================================
// js/event-state.js - Worker Terpusat (Otak Logika)
// ==========================================================
// Worker ini bertugas mencari event aktif, menghitung semua
// variabel waktu, dan mengirimkannya sebagai satu objek status.

let dailySchedule, clockOffset;

// Fungsi untuk menghitung waktu event berdasarkan string 'HH:mm:ss'
function getEventTime(baseDate, timeString) {
    const [hours, minutes, seconds] = timeString.split(':').map(Number);
    const eventDate = new Date(baseDate);
    eventDate.setHours(hours, minutes, seconds, 0);
    return eventDate.getTime();
}

function updateAndSendState() {
    if (!dailySchedule) return;

    const nowMs = Date.now() + clockOffset;
    const now = new Date(nowMs);
    const currentTime = now.toTimeString().slice(0, 8);

    // Cari event yang sedang aktif
    const activeEvent = dailySchedule.find(event =>
        currentTime >= event.start && currentTime < event.end
    );

    // DEBUG: Log untuk troubleshooting
    if (activeEvent) {
        console.log(`[EVENT WORKER] Active event found: ${activeEvent.mode} - ${activeEvent.prayer} (${activeEvent.start}-${activeEvent.end}) at ${currentTime}`);
    }

    if (activeEvent) {
        // Jika ada event aktif, hitung semuanya di sini
        const startTime = getEventTime(now, activeEvent.start);
        const endTime = getEventTime(now, activeEvent.end);
        const totalDuration = (endTime - startTime) / 1000;
        const elapsed = (nowMs - startTime) / 1000;
        const remainingTime = Math.max(0, totalDuration - elapsed);
        const progressPercentage = totalDuration > 0 ? Math.min(100, (elapsed / totalDuration) * 100) : 0;
        const currentHour = now.getHours();
        
        const state = {
            isActive: true,
            mode: activeEvent.mode,
            prayer: activeEvent.prayer,
            remainingTime: remainingTime,
            progress: progressPercentage,
            time: {
                now: now.toTimeString().slice(0,8),
                date: now.toLocaleDateString('id-ID', { weekday: 'long', year: 'numeric', month: 'long', day: 'numeric' }),
                hour: currentHour
            }
        };

        // DEBUG: Log state for iqomah specifically
        if (activeEvent.mode === 'iqomah') {
            console.log(`[EVENT WORKER] Iqomah state:`, state);
        }

        // Kirim objek status yang lengkap
        postMessage(state);

    } else {
        // Jika tidak ada event, kirim status tidak aktif
        postMessage({ isActive: false });
    }
}

// Terima data awal dari halaman
onmessage = (e) => {
    if (e.data.dailySchedule) {
        dailySchedule = e.data.dailySchedule;
        clockOffset = e.data.clockOffset || 0;

        console.log(`[EVENT WORKER] Initialized with ${dailySchedule.length} events, clockOffset: ${clockOffset}`);
        
        // DEBUG: Log iqomah events specifically
        const iqomahEvents = dailySchedule.filter(event => event.mode === 'iqomah');
        console.log(`[EVENT WORKER] Found ${iqomahEvents.length} iqomah events:`, iqomahEvents);

        // Jalankan sekali saat inisialisasi, lalu setiap detik
        updateAndSendState();
        setInterval(updateAndSendState, 1000);
    }
};
