// iqomah-layer.js - Iqomah Layer Management for Main.php
// ======================================================

class IqomahLayerManager {
    constructor() {
        console.log("IqomahLayerManager constructor called");
        console.log("Initializing elements...");
        
        this.elements = {
            prayerInfo: document.getElementById('prayerInfo'),
            countdownTime: document.getElementById('countdownTime'),
            nasihatItem: document.getElementById('nasihatItem')
        };
        
        console.log("Elements found:", {
            prayerInfo: !!this.elements.prayerInfo,
            countdownTime: !!this.elements.countdownTime,
            nasihatItem: !!this.elements.nasihatItem
        });
        
        this.isActive = false;
        this.currentState = null;
        this.currentNasihatIndex = 0;
        this.nasihatInterval = null;
        this.nasihatWaitInterval = null;
        this.nasihatInitialized = false;
        
        // Debug: Check nasihatList availability immediately
        console.log("nasihatList check in constructor:", typeof window.nasihatList !== 'undefined' ? window.nasihatList : 'UNDEFINED');
    }
    
    activate(state) {
        if (!state) {
            console.log("No state provided to IqomahLayerManager");
            return;
        }
        
        console.log("Activating iqomah layer with state:", state);
        this.isActive = true;
        this.currentState = state;

        // Update prayer name
        if (this.elements.prayerInfo) {
            this.elements.prayerInfo.textContent = state.prayer || "Iqomah";
        }

        // Update countdown
        this.updateCountdown(state);

        // Start nasihat rotation if not already running and not initialized
        if (this.elements.nasihatItem && !this.nasihatInitialized) {
            console.log("Starting nasihat rotation...");
            this.waitForNasihatList();
        } else {
            console.log("Nasihat not started - element:", !!this.elements.nasihatItem, "initialized:", this.nasihatInitialized);
        }

        // Update body classes for visual effects
        this.updateBodyClasses(state);

        // Check for completion
        if (state.remainingTime <= 0) {
            console.log("Iqomah countdown finished, redirecting to blank screen");
            window.location.href = "blank-screen.php";
            return;
        }
    }

    waitForNasihatList() {
        // Prevent multiple polling instances
        if (this.nasihatWaitInterval) {
            console.log("⚠️ Nasihat polling already running, skipping...");
            return;
        }
        
        console.log("⏳ Waiting for nasihatList to be available...");
        
        // Check immediately first
        if (typeof window.nasihatList !== 'undefined' && window.nasihatList.length > 0) {
            this.initializeNasihatRotation();
            return;
        }
        
        // If not available, poll every 100ms for up to 5 seconds
        let attempts = 0;
        const maxAttempts = 50; // 50 * 100ms = 5 seconds
        
        this.nasihatWaitInterval = setInterval(() => {
            attempts++;
            console.log(`⏳ Attempt ${attempts}/${maxAttempts} - checking nasihatList...`);
            
            if (typeof window.nasihatList !== 'undefined' && window.nasihatList.length > 0) {
                console.log("✅ nasihatList found! Initializing...");
                clearInterval(this.nasihatWaitInterval);
                this.nasihatWaitInterval = null;
                this.initializeNasihatRotation();
                return;
            }
            
            if (attempts >= maxAttempts) {
                console.log("❌ Timeout waiting for nasihatList");
                clearInterval(this.nasihatWaitInterval);
                this.nasihatWaitInterval = null;
                
                // Fallback: use hardcoded nasihat
                this.useFallbackNasihat();
            }
        }, 100);
    }

    useFallbackNasihat() {
        console.log("🔄 Using fallback nasihat...");
        
        // Define fallback nasihat directly
        const fallbackNasihat = [
            "Matikan atau silent handphone Anda",
            "Isi shaf terdepan terlebih dahulu", 
            "Luruskan dan rapatkan shaf",
            "Perbanyak doa antara adzan dan iqomah",
            "Dianjurkan sholat sunnah sebelum iqomah",
            "Tenangkan hati dan fokuskan pikiran"
        ];
        
        // Set fallback to window
        window.nasihatList = fallbackNasihat;
        this.initializeNasihatRotation();
    }

    initializeNasihatRotation() {
        // Prevent multiple initialization
        if (this.nasihatInitialized) {
            console.log("⚠️ Nasihat already initialized, skipping...");
            return;
        }
        
        console.log("initializeNasihatRotation called");
        console.log("window.nasihatList:", typeof window.nasihatList !== 'undefined' ? window.nasihatList : 'UNDEFINED');
        
        if (!window.nasihatList || window.nasihatList.length === 0) {
            console.log("❌ No nasihat list available");
            return;
        }

        console.log("✅ Initializing nasihat rotation with", window.nasihatList.length, "items");
        
        // Mark as initialized to prevent duplicate
        this.nasihatInitialized = true;
        
        // Clear any existing interval first
        if (this.nasihatInterval) {
            clearInterval(this.nasihatInterval);
            this.nasihatInterval = null;
        }
        
        // Check if nasihat element already has content from PHP
        if (this.elements.nasihatItem && this.elements.nasihatItem.textContent.trim()) {
            console.log("✅ Default nasihat already loaded from PHP:", this.elements.nasihatItem.textContent);
            // Find the index of current nasihat in the list (if exists)
            const currentText = this.elements.nasihatItem.textContent.trim();
            const foundIndex = window.nasihatList.findIndex(item => item === currentText);
            if (foundIndex !== -1) {
                this.currentNasihatIndex = foundIndex;
                console.log("✅ Found current nasihat at index:", foundIndex);
            }
        } else {
            // Fallback: display first nasihat if no default content
            this.displayNasihat();
        }
        
        // Start interval for rotation (8 seconds = 8000ms)
        this.nasihatInterval = setInterval(() => {
            if (this.isActive) { // Only update if layer is still active
                this.updateNasihat();
            }
        }, 8000);
        
        console.log("✅ Nasihat interval started with 8 second interval");
    }

    displayNasihat() {
        // Display current nasihat
        if (this.elements.nasihatItem && window.nasihatList && window.nasihatList.length > 0) {
            this.elements.nasihatItem.textContent = window.nasihatList[this.currentNasihatIndex];
            this.elements.nasihatItem.classList.add('active');
            console.log("✅ Displayed nasihat:", window.nasihatList[this.currentNasihatIndex]);
        }
    }

    updateNasihat() {
        if (!window.nasihatList || window.nasihatList.length === 0 || !this.elements.nasihatItem) {
            console.log("❌ Cannot update nasihat - missing requirements");
            return;
        }

        console.log("🔄 Updating nasihat, current index:", this.currentNasihatIndex);
        
        this.elements.nasihatItem.classList.remove('active');
        setTimeout(() => {
            this.currentNasihatIndex = (this.currentNasihatIndex + 1) % window.nasihatList.length;
            const nasihatText = window.nasihatList[this.currentNasihatIndex];
            this.elements.nasihatItem.textContent = nasihatText;
            this.elements.nasihatItem.classList.add('active');
            console.log("✅ Updated nasihat to:", nasihatText);
        }, 500);
    }

    updateCountdown(state) {
        if (!this.elements.countdownTime || !state) return;

        const minutes = Math.floor(state.remainingTime / 60);
        const seconds = Math.floor(state.remainingTime % 60);
        const timeText = `${minutes.toString().padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;

        this.elements.countdownTime.textContent = timeText;
    }

    updateBodyClasses(state) {
        const body = document.body;
        if (!body || !state) return;

        // Remove existing mode classes
        body.classList.remove('urgent-mode', 'beep-mode');

        // Add appropriate classes based on remaining time
        if (state.remainingTime <= 20 && state.remainingTime > 5) {
            body.classList.add('urgent-mode');
        } else if (state.remainingTime <= 5 && state.remainingTime > 0) {
            body.classList.add('beep-mode');
        }
    }

    deactivate() {
        console.log("Deactivating iqomah layer");
        this.isActive = false;
        this.currentState = null;

        // Clear nasihat interval
        if (this.nasihatInterval) {
            clearInterval(this.nasihatInterval);
            this.nasihatInterval = null;
            console.log("✅ Nasihat interval cleared");
        }

        // Clear nasihat wait interval
        if (this.nasihatWaitInterval) {
            clearInterval(this.nasihatWaitInterval);
            this.nasihatWaitInterval = null;
            console.log("✅ Nasihat wait interval cleared");
        }

        // Reset initialization flag
        this.nasihatInitialized = false;

        // Remove body classes
        const body = document.body;
        if (body) {
            body.classList.remove('urgent-mode', 'beep-mode');
        }

        // Reset elements
        if (this.elements.prayerInfo) {
            this.elements.prayerInfo.textContent = "Memuat...";
        }
        if (this.elements.countdownTime) {
            this.elements.countdownTime.textContent = "--:--";
        }
        if (this.elements.nasihatItem) {
            this.elements.nasihatItem.textContent = "";
            this.elements.nasihatItem.classList.remove('active');
        }
    }

    isCurrentlyActive() {
        return this.isActive;
    }

    getCurrentState() {
        return this.currentState;
    }
}

// Make IqomahLayerManager available globally
window.IqomahLayerManager = IqomahLayerManager;

console.log("IqomahLayerManager class defined and made available globally");
