#!/usr/bin/env python3
"""
Script monitoring diagnostik untuk folder ringan
Setiap kali file berubah, snapshot akan disimpan.
"""
import os
import hashlib
import json
import time
import fcntl
import shutil
from datetime import datetime

# Konfigurasi untuk folder ringan
WATCH_FILES = [
    'main.php',
    'adzan-display.php',
    'iqomah-countdown.php',
    'blank-screen.php',
    'main.js',
    'adzan-display.js',
    'iqomah-countdown.js',
    'blank-screen.js',
    'event.js',
    'countdown.js',
    'clock.js',
    'slide.js',
    'css/main.css',
    'css/adzan-display.css',
    'css/iqomah-countdown.css',
    'css/blank-screen.css',
    'index.php'
]

BASE_DIR = '/mmj/ringan'
HASH_FILE = os.path.join(BASE_DIR, 'data/file_hashes_ringan.json')
TIMESTAMP_FILE = os.path.join(BASE_DIR, 'data/last_change_ringan.json')
LOCK_FILE = os.path.join(BASE_DIR, 'data/monitor_ringan.lock')
SNAPSHOT_DIR = os.path.join(BASE_DIR, 'data/snapshots')

def get_file_hash(filepath):
    try:
        with open(filepath, 'rb') as f:
            return hashlib.md5(f.read()).hexdigest()
    except (FileNotFoundError, PermissionError):
        return None

def load_previous_hashes():
    if os.path.exists(HASH_FILE):
        try:
            with open(HASH_FILE, 'r') as f:
                return json.load(f)
        except:
            pass
    return {}

def save_hashes(hashes):
    try:
        os.makedirs(os.path.dirname(HASH_FILE), exist_ok=True)
        temp_file = HASH_FILE + '.tmp'
        with open(temp_file, 'w') as f:
            json.dump(hashes, f, indent=2)
        os.rename(temp_file, HASH_FILE)
    except Exception as e:
        print(f"Error saving hashes: {e}")

def update_timestamp():
    try:
        os.makedirs(os.path.dirname(TIMESTAMP_FILE), exist_ok=True)
        timestamp_data = {
            'last_change': int(time.time()),
            'last_change_readable': datetime.now().isoformat(),
            'status': 'changed'
        }
        temp_file = TIMESTAMP_FILE + '.tmp'
        with open(temp_file, 'w') as f:
            json.dump(timestamp_data, f, indent=2)
        os.rename(temp_file, TIMESTAMP_FILE)
        print(f"RINGAN: Timestamp updated: {timestamp_data['last_change_readable']}")
    except Exception as e:
        print(f"Error updating timestamp: {e}")

def main():
    try:
        os.makedirs(os.path.dirname(LOCK_FILE), exist_ok=True)
        lock_fd = open(LOCK_FILE, 'w')
        fcntl.flock(lock_fd, fcntl.LOCK_EX | fcntl.LOCK_NB)
    except IOError:
        print("Another ringan diagnostic instance is running, exiting...")
        return

    try:
        os.makedirs(SNAPSHOT_DIR, exist_ok=True)
        print(f"Diagnostic monitoring RINGAN started at {datetime.now().isoformat()}")
        
        previous_hashes = load_previous_hashes()
        current_hashes = {}
        has_changes = False

        for file_path in WATCH_FILES:
            full_path = os.path.join(BASE_DIR, file_path)
            current_hash = get_file_hash(full_path)

            if current_hash is not None:
                current_hashes[file_path] = current_hash
                previous_hash = previous_hashes.get(file_path)

                if previous_hash != current_hash:
                    print(f"!!! Change detected in: {file_path}")
                    print(f"    Old hash: {previous_hash}")
                    print(f"    New hash: {current_hash}")
                    has_changes = True

                    # Simpan snapshot!
                    timestamp_str = datetime.now().strftime('%Y%m%d_%H%M%S')
                    snapshot_name = f"{os.path.basename(file_path)}.{timestamp_str}.bak"
                    snapshot_path = os.path.join(SNAPSHOT_DIR, snapshot_name)
                    shutil.copy2(full_path, snapshot_path)
                    print(f"    SNAPSHOT SAVED to: {snapshot_path}")
        
        save_hashes(current_hashes)
        
        if has_changes:
            update_timestamp()
        else:
            print("No changes detected this run.")

    finally:
        fcntl.flock(lock_fd, fcntl.LOCK_UN)
        lock_fd.close()

if __name__ == '__main__':
    main()
