<?php
// index.php - Main container for iframe loading main.php and other displays with server failover
// =======================================================================

require_once 'conf.php';
?>
<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <link rel="shortcut icon" href="favicon.ico" type="image/x-icon">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo "$masjid_name"; ?> - Tampilan Utama</title>
    <style>
        /* Menghapus margin dan padding default dari browser */
        html, body {
            margin: 0;
            padding: 0;
            width: 100%;
            height: 100%;
            overflow: hidden; /* Mencegah scrollbar muncul */
            cursor: none;
            border: none;
        }

        body {
            position: relative;
            background-color: #000000;
            container-type: size;
        }

        #content-container {
            border: none;
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            aspect-ratio: 16 / 9;
            background-color: #000000;
            display: flex;
            justify-content: center;
            align-items: center;
            text-align: center;
        }
        
        @container (min-aspect-ratio: 16 / 9) {
            #content-container {
                height: 100%;
            }
        }

        @container (max-aspect-ratio: 16 / 9) {
            #content-container {
                width: 100%;
            }
        }

        #mainFrame {
            width: 100%;
            height: 100%;
            border: none;
            display: block;
            cursor: none;
            background-color: #000000;
        }
        #preload { display: none; }

        /* === CSS UNTUK INDIKATOR KONEKSI === */
        #connection-indicator {
            position: fixed;
            top: 15px;
            right: 15px;
            width: 15px;
            height: 15px;
            border-radius: 50%;
            background-color: #ffc107;
            box-shadow: 0 0 8px rgba(0, 0, 0, 0.5);
            transition: background-color 0.5s ease;
            z-index: 9999;
    	    opacity: 0.8;
            display: none;
        }

        #connection-indicator.status-good {
            background-color: #28a745; /* Hijau */
        }

        #connection-indicator.status-degraded {
            background-color: #ffc107; /* Oranye */
        }

        #connection-indicator.status-disconnected {
            background-color: #dc3545; /* Merah */
        }

        /* === CSS UNTUK INDIKATOR REFRESH === */
        #refresh-indicator {
            position: fixed;
            top: 15px;
            right: 40px;
            width: 12px;
            height: 12px;
            border-radius: 50%;
            background-color: #17a2b8;
            box-shadow: 0 0 8px rgba(0, 0, 0, 0.5);
            opacity: 0;
            transition: opacity 0.3s ease;
            z-index: 9999;
        }

        #refresh-indicator.active {
            opacity: 0.8;
        }

        /* === CSS UNTUK INDIKATOR STATUS MONITORING === */
        #monitor-status {
            position: fixed;
            top: 1cqh;
            right: 1cqw;
            width: 1cqw;
            height: 1cqw;
            border-radius: 50%;
            background-color: #6c757d; /* Abu-abu default */
            box-shadow: 0 0 8px rgba(0, 0, 0, 0.5);
            opacity: 0.6;
            transition: opacity 0.3s ease;
            z-index: 9999;
        }

        #monitor-status.active {
            background-color: #007bff; /* Biru - monitoring aktif */
            opacity: 0.8;
        }

        #monitor-status.paused {
            background-color: #fd7e14; /* Oranye - monitoring dijeda */
            opacity: 0.8;
        }

        #monitor-status.error {
            background-color: #dc3545; /* Merah - error berulang */
            opacity: 0.8;
        }

    </style>
</head>
<body>

    <div id="connection-indicator" title="Status Koneksi"></div>
    <div id="refresh-indicator" title="Auto-refresh Active"></div>
    <div id="monitor-status" title="Status Monitoring"></div>

    <div id="content-container">
        <iframe id="mainFrame" src="main.php" frameborder="0" allowfullscreen></iframe>
    </div>

<script>
document.addEventListener('DOMContentLoaded', () => {

    const indicator = document.getElementById('connection-indicator');
    const refreshIndicator = document.getElementById('refresh-indicator');
    const monitorStatus = document.getElementById('monitor-status');
    const mainFrame = document.getElementById('mainFrame');

    // Konfigurasi Pengecekan
    const PING_URL = 'ping.txt';
    const CHANGE_CHECK_URL = 'check_changes.php';
    const GOOD_THRESHOLD = 2500; // 2.5 detik
    const DEGRADED_THRESHOLD = 5000; // 5 detik
    
    // Konfigurasi untuk Change Monitoring - DIPERBAIKI
    const CHANGE_CHECK_TIMEOUT = 4000; // 4 detik timeout (dikurangi dari 8 detik)
    const MAX_CONSECUTIVE_FAILURES = 3; // Dikurangi dari 5 ke 3
    const BACKOFF_MULTIPLIER = 1.5; // Dikurangi dari 2 ke 1.5
    const MIN_INTERVAL = 60000; // 1 menit minimal
    const MAX_INTERVAL = 300000; // 5 menit maksimal (dikurangi dari 10 menit)
    const RECOVERY_CHECK_INTERVAL = 120000; // 2 menit (dikurangi dari 5 menit)

    let lastStatus = null;
    let lastChangeTimestamp = 0;
    let changeCheckInterval = null;
    let recoveryCheckInterval = null;
    
    // State untuk exponential backoff
    let consecutiveFailures = 0;
    let currentInterval = MIN_INTERVAL;
    let isMonitoringPaused = false;
    
    // PERBAIKAN: Tambahkan flag untuk mencegah request overlap
    let changeRequestInProgress = false;
    let connectionRequestInProgress = false;

    // Fungsi untuk mengubah tampilan indikator
    function updateIndicator(status) {
        if (status === lastStatus) return;
        indicator.classList.remove('status-good', 'status-degraded', 'status-disconnected');
        indicator.classList.add(`status-${status}`);
        lastStatus = status;
    }

    // Fungsi untuk mengupdate status monitoring
    function updateMonitorStatus(status) {
        monitorStatus.classList.remove('active', 'paused', 'error');
        if (status) {
            monitorStatus.classList.add(status);
        }
    }

    // Fungsi inti untuk mengecek koneksi - DIPERBAIKI
    async function checkConnection() {
        if (connectionRequestInProgress) {
            return lastStatus || 'disconnected';
        }
        
        connectionRequestInProgress = true;
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), DEGRADED_THRESHOLD);

        const startTime = performance.now();
        try {
            const response = await fetch(`${PING_URL}?t=${new Date().getTime()}`, {
                method: 'HEAD',
                cache: 'no-store',
                signal: controller.signal
            });
            
            clearTimeout(timeoutId);
            connectionRequestInProgress = false;
            const latency = performance.now() - startTime;

            if (response.ok && latency < GOOD_THRESHOLD) {
                return 'good';
            } else {
                return 'degraded';
            }
        } catch (error) {
            clearTimeout(timeoutId);
            connectionRequestInProgress = false;
            return 'disconnected';
        }
    }

    // Fungsi untuk menghitung interval berikutnya dengan exponential backoff
    function calculateNextInterval() {
        if (consecutiveFailures === 0) {
            return MIN_INTERVAL;
        }
        
        const backoffInterval = MIN_INTERVAL * Math.pow(BACKOFF_MULTIPLIER, consecutiveFailures - 1);
        return Math.min(backoffInterval, MAX_INTERVAL);
    }

    // Fungsi untuk reset monitoring ke normal
    function resetMonitoring() {
        consecutiveFailures = 0;
        currentInterval = MIN_INTERVAL;
        isMonitoringPaused = false;
        changeRequestInProgress = false; // PERBAIKAN: Reset flag
        updateMonitorStatus('active');
        
        // Stop recovery check jika sedang berjalan
        if (recoveryCheckInterval) {
            clearInterval(recoveryCheckInterval);
            recoveryCheckInterval = null;
        }
        
        console.log('Monitoring reset to normal interval');
    }

    // Fungsi untuk menghentikan monitoring sementara
    function pauseMonitoring() {
        isMonitoringPaused = true;
        changeRequestInProgress = false; // PERBAIKAN: Reset flag
        updateMonitorStatus('paused');
        
        if (changeCheckInterval) {
            clearInterval(changeCheckInterval);
            changeCheckInterval = null;
        }
        
        // Mulai recovery check
        if (!recoveryCheckInterval) {
            recoveryCheckInterval = setInterval(checkRecovery, RECOVERY_CHECK_INTERVAL);
        }
        
        console.log(`Monitoring paused after ${consecutiveFailures} consecutive failures`);
    }

    // Fungsi untuk mengecek recovery - DIPERBAIKI
    async function checkRecovery() {
        if (changeRequestInProgress) {
            return;
        }
        
        console.log('Checking if service has recovered...');
        
        changeRequestInProgress = true;
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), CHANGE_CHECK_TIMEOUT);
        
        try {
            const response = await fetch(`${CHANGE_CHECK_URL}?t=${new Date().getTime()}`, {
                method: 'GET',
                cache: 'no-store',
                signal: controller.signal
            });
            
            clearTimeout(timeoutId);
            changeRequestInProgress = false;
            
            if (response.ok) {
                console.log('Service recovered! Resuming normal monitoring...');
                
                // ✅ PERBAIKAN: Refresh iframe segera setelah recovery
                console.log('Refreshing iframe after recovery...');
                refreshIndicator.classList.add('active');
                mainFrame.src = mainFrame.src;
                setTimeout(() => {
                    refreshIndicator.classList.remove('active');
                }, 2000);
                resetMonitoring();
                startChangeMonitoring(); // Restart monitoring
            }
        } catch (error) {
            clearTimeout(timeoutId);
            changeRequestInProgress = false;
            console.log('Service still not available, will check again later');
        }
    }

    // Fungsi untuk mengecek perubahan timestamp dengan error handling - DIPERBAIKI
    async function checkForChanges() {
        if (isMonitoringPaused || changeRequestInProgress) {
            return;
        }
        
        changeRequestInProgress = true;
        const controller = new AbortController();
        const timeoutId = setTimeout(() => controller.abort(), CHANGE_CHECK_TIMEOUT);
        
        try {
            const response = await fetch(`${CHANGE_CHECK_URL}?t=${new Date().getTime()}`, {
                method: 'GET',
                cache: 'no-store',
                signal: controller.signal
            });

            clearTimeout(timeoutId);
            changeRequestInProgress = false;

            if (!response.ok) {
                throw new Error(`HTTP ${response.status}`);
            }

            const data = await response.json();
            
            if (data.success && data.timestamp) {
                const currentTimestamp = parseInt(data.timestamp);
                
                // Reset consecutive failures karena berhasil
                if (consecutiveFailures > 0) {
                    console.log(`Service recovered after ${consecutiveFailures} failures`);
                    resetMonitoring();
                }
                
                // Jika timestamp berubah, refresh iframe
                if (lastChangeTimestamp > 0 && currentTimestamp > lastChangeTimestamp) {
                    console.log(`Content changed detected at ${data.readable_time}. Refreshing iframe...`);
                    
                    refreshIndicator.classList.add('active');
                    mainFrame.src = mainFrame.src;
                    
                    setTimeout(() => {
                        refreshIndicator.classList.remove('active');
                    }, 2000);
                }
                
                lastChangeTimestamp = currentTimestamp;
                console.log(`Change check completed. Last change: ${data.readable_time}`);
                
            } else {
                throw new Error(data.error || 'Invalid response format');
            }
        } catch (error) {
            clearTimeout(timeoutId);
            changeRequestInProgress = false;
            consecutiveFailures++;
            
            console.warn(`Change check failed (${consecutiveFailures}/${MAX_CONSECUTIVE_FAILURES}):`, error.message);
            
            // Jika sudah terlalu banyak gagal, hentikan monitoring
            if (consecutiveFailures >= MAX_CONSECUTIVE_FAILURES) {
                updateMonitorStatus('error');
                pauseMonitoring();
                return;
            }
            
            // Update interval dengan exponential backoff
            currentInterval = calculateNextInterval();
            updateMonitorStatus('error');
            
            // Restart monitoring dengan interval yang lebih lama
            if (changeCheckInterval) {
                clearInterval(changeCheckInterval);
                changeCheckInterval = setInterval(checkForChanges, currentInterval);
                console.log(`Monitoring interval increased to ${currentInterval/1000} seconds`);
            }
        }
    }

    // Fungsi untuk memulai monitoring perubahan
    function startChangeMonitoring() {
        if (changeCheckInterval) {
            clearInterval(changeCheckInterval);
        }
        
        if (!isMonitoringPaused) {
            updateMonitorStatus('active');
            checkForChanges(); // Jalankan sekali langsung
            changeCheckInterval = setInterval(checkForChanges, currentInterval);
            console.log(`Change monitoring started with ${currentInterval/1000} seconds interval`);
        }
    }

    // Fungsi untuk menghentikan monitoring
    function stopChangeMonitoring() {
        if (changeCheckInterval) {
            clearInterval(changeCheckInterval);
            changeCheckInterval = null;
        }
        
        if (recoveryCheckInterval) {
            clearInterval(recoveryCheckInterval);
            recoveryCheckInterval = null;
        }
        
        // PERBAIKAN: Reset flag saat stop
        changeRequestInProgress = false;
        connectionRequestInProgress = false;
        
        updateMonitorStatus(null);
        console.log('Change monitoring stopped');
    }

    // Fungsi untuk pengecekan koneksi berkala - DIPERBAIKI
    async function periodicConnectionCheck() {
        const status = await checkConnection();
        updateIndicator(status);

        const currentMinutes = new Date().getMinutes();

        // Fallback refresh jika monitoring bermasalah
        if (status === 'good' && (currentMinutes % 30 === 0 || (isMonitoringPaused && currentMinutes % 10 === 0))) {
            console.log(`Fallback refresh at ${new Date().toLocaleTimeString()}`);
                refreshIndicator.classList.add('active');
                setTimeout(() => {
                    refreshIndicator.classList.remove('active');
                }, 2000);
            mainFrame.src = mainFrame.src;
        }
    }

    // === INISIALISASI ===
    
    // Jalankan pengecekan koneksi pertama kali
    
    // ✅ PERBAIKAN: Initial check - refresh iframe jika terdeteksi masalah koneksi
    setTimeout(async () => {
        const initialStatus = await checkConnection();
        if (initialStatus === 'disconnected') {
            console.log('Initial connection check failed, will retry iframe load in 5 seconds...');
            setTimeout(() => {
                console.log('Retrying iframe load after initial failure...');
                refreshIndicator.classList.add('active');
                mainFrame.src = mainFrame.src;
                setTimeout(() => {
                    refreshIndicator.classList.remove('active');
                }, 2000);
            }, 5000);
        }
    }, 1000);
    periodicConnectionCheck();
    
    // Mulai monitoring perubahan
    startChangeMonitoring();

    // Atur interval untuk pengecekan koneksi setiap 10 menit
    setInterval(periodicConnectionCheck, 600000);

    // Handle visibility change untuk mengoptimalkan resource - DIPERBAIKI
    document.addEventListener('visibilitychange', () => {
        if (document.hidden) {
            stopChangeMonitoring();
        } else {
            // Reset state dan mulai ulang jika sebelumnya ada masalah
            if (isMonitoringPaused) {
                resetMonitoring();
                // ✅ PERBAIKAN: Refresh iframe jika sebelumnya ada masalah
                console.log('Refreshing iframe after visibility change recovery...');
                refreshIndicator.classList.add('active');
                mainFrame.src = mainFrame.src;
                setTimeout(() => {
                    refreshIndicator.classList.remove('active');
                }, 2000);
            }
            startChangeMonitoring();
        }
    });

    // Handle beforeunload untuk cleanup - DIPERBAIKI
    window.addEventListener('beforeunload', () => {
        stopChangeMonitoring();
    });

    // Debug: Tampilkan status monitoring di console setiap 5 menit
    setInterval(() => {
        console.log(`Monitoring status: ${isMonitoringPaused ? 'PAUSED' : 'ACTIVE'}, Failures: ${consecutiveFailures}, Interval: ${currentInterval/1000}s, Request in progress: ${changeRequestInProgress}`);
    }, 300000);

});

    // ===============================================
    // 🔄 FALLBACK CONTENT SYSTEM - Enhanced Cache
    // ===============================================
</script>

</body>
</html>
========================================================================
    // ===============================================
    // 🔄 FALLBACK CONTENT SYSTEM - Enhanced Cache
    // ===============================================
    
    // Konfigurasi
    const FALLBACK_CONFIG = {
        CHECK_INTERVAL: 30000,     // 30 detik
        CACHE_TIMEOUT: 86400000,   // 24 jam (expired cache)
        CACHE_KEY: 'masjid_fallback_cache',
        DEBUG: true
    };

    let fallbackSystem = {
        cachedContent: null,
        cacheTimestamp: null,
        isUsingFallback: false,
        checkInterval: null,
        
        // Inisialisasi sistem fallback
        init: function() {
            this.loadFromCache();
            this.setupIframeMonitoring();
            this.startPeriodicCheck();
            this.log('Fallback system initialized');
        },
        
        // Load cache dari localStorage
        loadFromCache: function() {
            try {
                const cached = localStorage.getItem(FALLBACK_CONFIG.CACHE_KEY);
                if (cached) {
                    const data = JSON.parse(cached);
                    // Check if cache is not expired
                    if (Date.now() - data.timestamp < FALLBACK_CONFIG.CACHE_TIMEOUT) {
                        this.cachedContent = data.content;
                        this.cacheTimestamp = data.timestamp;
                        this.log('Cache loaded from localStorage', data.timestamp);
                    } else {
                        this.log('Cache expired, clearing...');
                        localStorage.removeItem(FALLBACK_CONFIG.CACHE_KEY);
                    }
                }
            } catch (error) {
                this.log('Error loading cache:', error);
            }
        },
        
        // Save content ke cache
        saveToCache: function(content) {
            try {
                const data = {
                    content: content,
                    timestamp: Date.now(),
                    url: window.location.href
                };
                localStorage.setItem(FALLBACK_CONFIG.CACHE_KEY, JSON.stringify(data));
                this.cachedContent = content;
                this.cacheTimestamp = data.timestamp;
                this.log('Content cached successfully');
            } catch (error) {
                this.log('Error saving cache:', error);
            }
        },
        
        // Setup monitoring iframe
        setupIframeMonitoring: function() {
            const iframe = document.getElementById('mainFrame');
            if (!iframe) return;
            
            // Event listener untuk iframe load
            iframe.addEventListener('load', () => {
                setTimeout(() => {
                    this.checkIframeContent();
                }, 1000); // Delay untuk memastikan content ter-load
            });
            
            // Event listener untuk iframe error
            iframe.addEventListener('error', () => {
                this.log('Iframe error detected');
                this.applyFallback();
            });
        },
        
        // Check content iframe
        checkIframeContent: function() {
            try {
                const iframe = document.getElementById('mainFrame');
                if (!iframe || !iframe.contentWindow) return;
                
                const iframeDoc = iframe.contentWindow.document;
                const body = iframeDoc.body;
                
                if (!body) {
                    this.log('Iframe body not accessible');
                    return;
                }
                
                const content = body.innerHTML;
                const isError = this.isErrorContent(content);
                
                if (isError) {
                    this.log('Error content detected in iframe');
                    this.applyFallback();
                } else if (content.trim().length > 100) { // Minimal content check
                    this.log('Valid content detected, updating cache');
                    this.saveToCache(content);
                    this.isUsingFallback = false;
                }
                
            } catch (error) {
                this.log('Cannot access iframe content (CORS/Security):', error.message);
                // Jika tidak bisa akses content iframe (CORS), gunakan fallback
                this.applyFallback();
            }
        },
        
        // Check apakah content adalah error
        isErrorContent: function(content) {
            if (!content || content.trim().length < 50) return true;
            
            const errorPatterns = [
                '404', 'not found', 'error', 'failed to load',
                'connection refused', 'timeout', 'unavailable',
                'page not found', 'server error'
            ];
            
            const lowerContent = content.toLowerCase();
            return errorPatterns.some(pattern => lowerContent.includes(pattern));
        },
        
        // Apply fallback content
        applyFallback: function() {
            if (!this.cachedContent) {
                this.log('No cached content available for fallback');
                this.showNoConnectionMessage();
                return;
            }
            
            try {
                const iframe = document.getElementById('mainFrame');
                if (iframe && iframe.contentWindow && iframe.contentWindow.document) {
                    iframe.contentWindow.document.body.innerHTML = this.cachedContent;
                    this.isUsingFallback = true;
                    this.log('Fallback content applied');
                    this.showFallbackIndicator();
                } else {
                    // Jika tidak bisa akses iframe, buat content div
                    this.createFallbackDiv();
                }
            } catch (error) {
                this.log('Error applying fallback:', error);
                this.createFallbackDiv();
            }
        },
        
        // Buat div fallback jika iframe tidak accessible
        createFallbackDiv: function() {
            const container = document.getElementById('content-container');
            if (!container) return;
            
            // Hide iframe
            const iframe = document.getElementById('mainFrame');
            if (iframe) iframe.style.display = 'none';
            
            // Create fallback div
            let fallbackDiv = document.getElementById('fallback-content');
            if (!fallbackDiv) {
                fallbackDiv = document.createElement('div');
                fallbackDiv.id = 'fallback-content';
                fallbackDiv.style.cssText = `
                    width: 100%;
                    height: 100%;
                    background: #000;
                    color: #fff;
                    overflow: hidden;
                `;
                container.appendChild(fallbackDiv);
            }
            
            fallbackDiv.innerHTML = this.cachedContent || this.getNoConnectionHTML();
            this.isUsingFallback = true;
            this.log('Fallback div created');
            this.showFallbackIndicator();
        },
        
        // Show "no connection" message
        showNoConnectionMessage: function() {
            this.createFallbackDiv();
            const fallbackDiv = document.getElementById('fallback-content');
            if (fallbackDiv) {
                fallbackDiv.innerHTML = this.getNoConnectionHTML();
            }
        },
        
        // HTML untuk "no connection"
        getNoConnectionHTML: function() {
            return `
                <div style="
                    display: flex;
                    flex-direction: column;
                    justify-content: center;
                    align-items: center;
                    height: 100%;
                    text-align: center;
                    font-family: Arial, sans-serif;
                ">
                    <div style="font-size: 48px; margin-bottom: 20px;">📡</div>
                    <h2 style="color: #ffc107; margin-bottom: 10px;">Koneksi Terputus</h2>
                    <p style="color: #ccc; margin-bottom: 20px;">Mencoba menghubungkan kembali...</p>
                    <div style="
                        width: 50px; 
                        height: 50px; 
                        border: 3px solid #333; 
                        border-top: 3px solid #ffc107; 
                        border-radius: 50%; 
                        animation: spin 1s linear infinite;
                    "></div>
                    <style>
                        @keyframes spin {
                            0% { transform: rotate(0deg); }
                            100% { transform: rotate(360deg); }
                        }
                    </style>
                </div>
            `;
        },
        
        // Show fallback indicator
        showFallbackIndicator: function() {
            let indicator = document.getElementById('fallback-indicator');
            if (!indicator) {
                indicator = document.createElement('div');
                indicator.id = 'fallback-indicator';
                indicator.style.cssText = `
                    position: fixed;
                    top: 15px;
                    left: 15px;
                    background: #ff6b6b;
                    color: white;
                    padding: 8px 12px;
                    border-radius: 15px;
                    font-size: 12px;
                    font-weight: bold;
                    z-index: 10000;
                    animation: pulse 2s infinite;
                `;
                indicator.innerHTML = '📄 Mode Offline';
                document.body.appendChild(indicator);
                
                // Add CSS animation
                const style = document.createElement('style');
                style.textContent = `
                    @keyframes pulse {
                        0% { opacity: 0.7; }
                        50% { opacity: 1; }
                        100% { opacity: 0.7; }
                    }
                `;
                document.head.appendChild(style);
            }
        },
        
        // Hide fallback indicator
        hideFallbackIndicator: function() {
            const indicator = document.getElementById('fallback-indicator');
            if (indicator) {
                indicator.remove();
            }
        },
        
        // Restore normal iframe
        restoreNormalMode: function() {
            // Show iframe
            const iframe = document.getElementById('mainFrame');
            if (iframe) iframe.style.display = 'block';
            
            // Hide fallback div
            const fallbackDiv = document.getElementById('fallback-content');
            if (fallbackDiv) fallbackDiv.style.display = 'none';
            
            this.hideFallbackIndicator();
            this.isUsingFallback = false;
            this.log('Restored to normal mode');
        },
        
        // Start periodic check
        startPeriodicCheck: function() {
            this.checkInterval = setInterval(() => {
                this.periodicCheck();
            }, FALLBACK_CONFIG.CHECK_INTERVAL);
            this.log('Periodic check started');
        },
        
        // Periodic check function
        periodicCheck: function() {
            this.log('Running periodic check...');
            
            // Check connection status
            const connectionStatus = lastStatus; // Dari sistem yang sudah ada
            
            if (connectionStatus === 'disconnected' || connectionStatus === 'degraded') {
                if (!this.isUsingFallback) {
                    this.log('Connection issue detected, applying fallback');
                    this.applyFallback();
                }
            } else if (connectionStatus === 'good' && this.isUsingFallback) {
                this.log('Connection restored, checking iframe content');
                setTimeout(() => {
                    this.checkIframeContent();
                    if (!this.isUsingFallback) {
                        this.restoreNormalMode();
                    }
                }, 2000);
            }
        },
        
        // Stop periodic check
        stop: function() {
            if (this.checkInterval) {
                clearInterval(this.checkInterval);
                this.checkInterval = null;
            }
            this.log('Fallback system stopped');
        },
        
        // Debug logging
        log: function(...args) {
            if (FALLBACK_CONFIG.DEBUG) {
                console.log('[FallbackSystem]', ...args);
            }
        }
    };

    // Auto-initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => {
            setTimeout(() => fallbackSystem.init(), 2000);
        });
    } else {
        setTimeout(() => fallbackSystem.init(), 2000);
    }

    // Cleanup on page unload
    window.addEventListener('beforeunload', () => {
        fallbackSystem.stop();
    });


